
:- module points.
:- interface.

:- import_module char, list.
:- import_module io.

:- type point
	--->	point(
				x :: int,
                y :: int,
				symbol :: char
			).

:- pred (point::in) < (point::in) is semidet.
:- pred (point::in) == (point::in) is semidet.
:- pred (point::in) > (point::in) is semidet.

:- type points_get
	--->	max_x ; max_y ; min_x ; min_y.

:- type points_set
	--->	add_x ; add_y ; rot_90; sort_min.

%
:- func chars_to_points(list(char), int, int) = list(point).
:- pred chars_to_points_l(list(char)::in, int::in, int::in,
							list(point)::in, list(point)::out) is det.

%
:- pred write_points(list(point)::in, io::di, io::uo) is det.

%
:- func points_cmp(point, point) = comparison_result.
:- func get_points_value(list(point), points_get) = int.
:- func set_points_value(list(point), points_set, int) = list(point).

:- implementation.

:- import_module int, string.

% compare operators for point
point(X1, Y1, _) < point(X2, Y2, _) :- ( Y1 < Y2 ; Y1 = Y2, X1 < X2 ).
point(X1, Y1, _) == point(X2, Y2, _) :- ( Y1 = Y2, X1 = X2 ).
P1 > P2 :- P2 < P1.

% convert line chars to points list (last point will be first)
chars_to_points(Chars, X, Y) = Points :-
	chars_to_points_l(Chars, X, Y, [], Points).

%
chars_to_points_l([], _, _, Points, Points).
chars_to_points_l([Char | Chars], X, Y, Points0, Points) :-
	char.to_int(Char, CharI),
	(
		(CharI >= 0x30, CharI =< 0x39 ; CharI >= 0x41, CharI =< 0x46)
	->
		% allow only 0-9, A-F
		Points1 = [point(X, Y, Char) | Points0]
	;
		Points1 = Points0
	),
	chars_to_points_l(Chars, X + 1, Y, Points1, Points).

% sort and write list of points to io
write_points(Points0, !IO) :-
	Points = set_points_value(Points0, sort_min, 0),
	write_points_0(Points, 0, 0, !IO).

:- pred write_points_0(list(point)::in, int::in, int::in,
						io::di, io::uo) is det.

write_points_0([], _, _, !IO).
write_points_0(Points0, Xo, Yo, !IO) :-
	Points0 = [point(X, Y, Char) | Points],
%	io.format("o(%d, %d) ", [i(Xo), i(Yo)], !IO),
%	io.format("p(%d, %d, '%c')", [i(X), i(Y), c(Char)], !IO),
%	write_points0(Points, Xo, Yo, !IO).
	(
		(X = Xo, Y = Yo)
	->
%		io.format("point(%d, %d, '%c')", [i(X), i(Y), c(Char)], !IO),
		io.write_char(Char, !IO),
		Xo1 = Xo + 1, Yo1 = Yo, Points1 = Points
	;
		(
			(X < Xo, Y =< Yo)
		->
			% invalid position (may be 2 equal points?)
			io.write_char('i', !IO),
			Xo1 = Xo, Yo1 = Yo, Points1 = Points
		;
			(
				Y > Yo
			->
				% new line
				io.nl(!IO),
				Xo1 = 0, Yo1 = Yo + 1, Points1 = Points0
			;
				% space
				io.write_char(' ', !IO),
				Xo1 = Xo + 1, Yo1 = Yo, Points1 = Points0
			)
		)
	),
	write_points_0(Points1, Xo1, Yo1, !IO).

% comparison func for sort and other
points_cmp(P1, P2) = R :-
	( P1 < P2 -> R = (<) ;
		( P1 > P2 -> R = (>) ; R = (=) )
	).

% get value from points list
get_points_value(Points, max_x) = list.foldl(points_max_x, Points, 0).
get_points_value(Points, max_y) = list.foldl(points_max_y, Points, 0).
get_points_value(Points, min_x) = list.foldl(points_min_x, Points, 0).
get_points_value(Points, min_y) = list.foldl(points_min_y, Points, 0).

:- func points_max_x(point, int) = int.
points_max_x(point(X,_,_), Xm0) = Xm :- (X > Xm0 -> Xm = X ; Xm = Xm0).

:- func points_max_y(point, int) = int.
points_max_y(point(_,Y,_), Ym0) = Ym :- (Y > Ym0 -> Ym = Y ; Ym = Ym0).

:- func points_min_x(point, int) = int.
points_min_x(point(X,_,_), Xi0) = Xi :- (X < Xi0 -> Xi = X ; Xi = Xi0).

:- func points_min_y(point, int) = int.
points_min_y(point(_,Y,_), Yi0) = Yi :- (Y < Yi0 -> Yi = Y ; Yi = Yi0).

% change value of each point in list (or sort list)
set_points_value(Points0, add_x, Value) = Points :-
					list.map_foldl(points_add_x, Points0, Points, Value, _).

set_points_value(Points0, add_y, Value) = Points :-
					list.map_foldl(points_add_y, Points0, Points, Value, _).

set_points_value(Points0, rot_90, _) = Points :-
					list.map_foldl(points_rot_90, Points0, Points, 0, _).

set_points_value(Points0, sort_min, _) = Points :-
					Points = list.sort(points_cmp, Points0).

:- pred points_add_x(point::in, point::out, int::in, int::out) is det.
points_add_x(point(X,Y,C), point(X + Value,Y,C), Value, Value).

:- pred points_add_y(point::in, point::out, int::in, int::out) is det.
points_add_y(point(X,Y,C), point(X,Y + Value,C), Value, Value).

:- pred points_rot_90(point::in, point::out, int::in, int::out) is det.
points_rot_90(point(X,Y,C), point(X,Y,C), _, 0).
