
/**
 *
 * @author gloom@opcode.cc 2017
 */

package org.mercury.lang;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import javax.swing.text.Document;
import org.netbeans.api.languages.ASTNode;
import org.netbeans.api.languages.CompletionItem;
import org.netbeans.api.languages.ParseException;
import org.netbeans.api.languages.ParserManager;
import org.netbeans.api.lexer.Token;
import org.netbeans.api.lexer.TokenHierarchy;
import org.netbeans.api.lexer.TokenSequence;
import org.openide.filesystems.FileObject;
import org.openide.util.Exceptions;

public class ParserHelper
{
	protected static final Pattern pattern_comment1 = Pattern.compile("/\\*.*\\*/");
	protected static final Pattern pattern_comment2 = Pattern.compile("%.*(?=\\n)"); // "//.*(?=\\n)"
	protected static final Pattern pattern_clean1   = Pattern.compile("(\\s)+");
	protected static final Pattern pattern_clean2   = Pattern.compile(",!(\\s)");

	// prepare line for parsing
	// XXX TODO func("/** I am not a comment */") -> func("")
	public static String cleanLine (String line)
	{
		// remove comments
		line = pattern_comment2.matcher(
				pattern_comment1.matcher(line).replaceAll("")).
					replaceAll("");//.trim();

		// XXX TODO slow, rewrite this regexp
//		line = line.replaceAll("(\\s)+", " ").replaceAll(",", ", ").trim();
		line = pattern_clean2.matcher(
				pattern_clean1.matcher(line).replaceAll(" ")).
					replaceAll(", ").trim();

		return line;
	}

	// return module name from declaration in node string or ""
	public static String getModuleName (String sNode) {
		return getModuleName(sNode, false);
	}

	public static String getModuleName (String sNode, boolean cleared)
	{
		// search for module declaration and parse it
		//
		// :- module xxxx.
		// :- module (xx_xx) .
		// :- module (thread.channel).
		// :- module ((mdbcomp.feedback).automatic_parallelism).
		//

		if (!sNode.trim().startsWith(":-") || sNode.indexOf("module") == -1)
			return "";

		if (!cleared)
			sNode = cleanLine(sNode);

		int dpos = sNode.lastIndexOf('.');
		int mpos = sNode.indexOf(" module ");
		if (mpos == -1 || mpos > dpos)
			return "";

		String sName = sNode.substring(mpos + 8, dpos).
						replaceAll("[()]", "").trim();
		return sName;
	}

	// return imports names from node string or ""
	public static String[] getImportNames (String sNode) {
		return getImportNames(sNode, false);
	}

	public static String[] getImportNames (String sNode, boolean cleared)
	{
		String[] imports = new String[0];

		// search for imports declaration and parse it
		//
		// :- import_module xxx1, xxx2.
		// :- import_module ((mdbcomp.feedback).automatic_parallelism).
		//

		if (!sNode.trim().startsWith(":-") || sNode.indexOf("import_module") == -1)
			return imports;

		if (!cleared)
			sNode = cleanLine(sNode);

		int dpos = sNode.lastIndexOf('.');
		int ipos = sNode.indexOf(" import_module ");
		if (ipos == -1 || ipos > dpos)
			return imports;

		imports = sNode.substring(ipos + 15, dpos).
					replaceAll("[()]", "").trim().split(",");
		return imports;
	}

	// return type name from declaration in node string or ""
	public static String getTypeName (String sNode) {
		return getTypeName(sNode, false);
	}

	public static String getTypeName (String sNode, boolean cleared)
	{
		// search for type declaration and parse it
		//
		// :- type array(T).
		// :- type index_out_of_bounds ---> index_out_of_bounds(string).
		// :- type dir.foldl_pred(T) == pred(string, string, io.file_type, bool, T, T, io, io).
		// :- type (A, B) ---> (A, B).
		//
		// TODO XXX add support?
		// :- typeclass duplex(Stream, State) <= (stream.input(Stream, State), stream.output(Stream, State)) where [ ].
		// :- pragma type_spec((map.det_update)/4, (K = int), (map.'TypeSpecOf__pred_or_func__det_update__[K = int]')).
		//

		if (!sNode.trim().startsWith(":-") || sNode.indexOf("type") == -1)
			return "";

		if (!cleared)
			sNode = cleanLine(sNode);

		// XXX TODO make normal check
		int epos = -1;
		int dpos = sNode.indexOf(" type ");
		if (dpos != -1)
		{
			epos = sNode.indexOf("--->");
			if (epos == -1)
			{
				epos = sNode.indexOf("==");
				if (epos == -1)
					epos = sNode.lastIndexOf('.');
			}
		}

		if (dpos == -1 || dpos >= epos)
			return "";

		// ok, type declaration
		String sName = sNode.substring(dpos + 6, epos).trim();
		return sName;
	}

	// return func or pred name from foreign proc implementation in node string or ""
	public static String getForeignProcName (String sNode) {
		return getForeignProcName(sNode, false);
	}

	public static String getForeignProcName (String sNode, boolean cleared)
	{
		// search for foreign proc implementation and parse it
		//
		// :- pragma foreign_proc("C#",
		//     dir.split_name_dotnet(FileName::in, DirName::out, BaseName::out),
		//     [will_not_call_mercury, promise_pure, thread_safe],
		// "
		//
		// :- pragma foreign_proc("C",
		//     is_ground(Var::ia, Result::out) /* cc_multi */,
		//     [promise_pure, may_call_mercury],
		// "
		//
		// :- pragma foreign_proc("C",
		//     can_implement_make_directory,
		//     [will_not_call_mercury, promise_pure, thread_safe, will_not_modify_trail,
		//         does_not_affect_liveness],
		// "
		//
		// :- pragma foreign_proc("C",
		//     var(Value::in) = (Var::out) /* det */,
		//     [promise_pure, may_call_mercury],
		// "
		//
		// XXX TODO support
		//
		// :- pragma c_code( var(Value::(free -> clobbered_any)) = (Var::oa), % det
		//     may_call_mercury,
		// "
		//

		if (!sNode.trim().startsWith(":-") || sNode.indexOf("foreign_proc") == -1)
			return "";

		if (!cleared)
			sNode = cleanLine(sNode);

		int dpos = sNode.indexOf(" foreign_proc ");
		if (dpos == -1)
		{
			dpos = sNode.indexOf(" foreign_proc(");
			if (dpos == -1)
				return "";
		}

		int epos = sNode.indexOf("[", dpos); // XXX TODO can be in pred decl?
		if (epos != -1)
			epos = sNode.lastIndexOf(",", epos);
		dpos = sNode.indexOf(',', dpos);

		if (dpos == -1 || dpos >= epos)
			return "";

		// ok, func or pred declaration
		String sName = sNode.substring(dpos + 1, epos).trim();

		// XXX TODO parse name with getMethodName
		boolean isFunc = (sName.indexOf('=') != -1 && sName.indexOf("==") == -1);
		sName = getMethodName((isFunc ? ":- func " : ":- pred ") + sName + ".", true);

		return sName;
	}

	// return func or pred name from declaration in node string or ""
	public static String getMethodName (String sNode) {
		return getMethodName(sNode, false);
	}

	public static String getMethodName (String sNode, boolean cleared)
	{
		// search for func or pred declaration and parse it
		//
		// :- func var(T) = var(T).
		// :- func var.rep_free = (var_rep(T)::out(var_rep_any)) is det.
		// :- func var.rep_alias(var_rep(T)::in(var_rep_any)) = (var_rep(T)::out(var_rep_any)) is det.
		// :- pred var(T) == var(T).
		// :- impure pred destructively_update_binding(var_rep(T), var_rep(T)).
		// :- pred var.rep_init_with_value(T::in, var_rep(T)::out(ptr(var_rep_ground))) is det.
		// :- pred write(int, T, io.io, io.io) <= (pprint.doc(T)).
		//
		// :- func (mkEncoding(Enc) = encoding) <= encoding(Enc).
		// :- func 'byte :='(bitmap.byte_index, bitmap.bitmap, bitmap.byte) = bitmap.bitmap.
		// :- func foldl((func(L, A) = A), list.list(L), A) = A.
		//

		if (!sNode.trim().startsWith(":-") || (
			sNode.indexOf("pred") == -1 && sNode.indexOf("func") == -1))
		{
			return "";
		}

		if (!cleared)
			sNode = cleanLine(sNode);

		// XXX TODO make normal check
		int epos;
		int dpos = sNode.indexOf(" pred ");
		if (dpos == -1)
		{
			dpos = sNode.indexOf(" func ");
			epos = sNode.lastIndexOf("=");

			// XXX
			int tpos = sNode.lastIndexOf("<=");
			if (tpos == -1)
				tpos = sNode.lastIndexOf("=>");
			if (tpos != -1)
				epos = sNode.substring(0, tpos).lastIndexOf("=");
		}
		else
		{
			// pred
			epos = sNode.lastIndexOf("<=");
			if (epos == -1)
				epos = sNode.lastIndexOf('.');
		}

		if (dpos == -1 || dpos >= epos)
			return "";

		// ok, func or pred declaration
		String sName = sNode.substring(dpos + 6, epos).trim();

		// remove 'is multi' or other at end
		int ipos = sName.lastIndexOf(" is ", sName.length());
		if (ipos != -1)
			sName = sName.substring(0, ipos).trim();

		return sName;
	}

	// -----------------------------------------------------------------

	// replace io parameters with !IO
	public static String fixMethodNameIO (String name)
	{
		// replace with !IO
		//
		// io.io, io.io)
		// io::di, io::uo)
		// io, io)
		//

		//int ipos = -1;
		if (name.endsWith("io.io, io.io)"))
		{
			//ipos = name.lastIndexOf("io.io, io.io)");
			//name = name.replaceFirst("io.io, io.io\\)", "!IO)");
			name = name.replaceAll("io.io, io.io\\)", "!IO)");
		}
		else if (name.endsWith("io, io)"))
		{
			//ipos = name.lastIndexOf("io, io)");
			name = name.replaceAll("io, io\\)", "!IO)");
		}
		else if (name.endsWith("io::di, io::uo)"))
		{
			//ipos = name.lastIndexOf("io::di, io::uo)");
			name = name.replaceAll("io::di, io::uo\\)", "!IO)");
		}
		else if (name.endsWith("State, State)"))
		{
			name = name.replaceAll("State, State\\)", "!State)");
		}
		else if (name.endsWith("State::di, State::uo)"))
		{
			//ipos = name.lastIndexOf("State::di, State::uo)");
			name = name.replaceAll("State::di, State::uo\\)", "!State)");
		}
		else
		{
			return "";
		}

		//if (ipos == -1)
		//	return "";
		//return name.substring(0, ipos) + "!IO)";
		return name;
	}

	// get method name without params
	public static String getMethodShortName (String name)
	{
		int bpos = name.indexOf("(");
		if (bpos == -1)
			return "";

		return name.substring(0, bpos);
	}

	// -----------------------------------------------------------------

	// get root ASTNode for document from ParserManager
	public static ASTNode getDocumentAST (Document document)
	{
		ASTNode node = null;
		int counter = 0;

		// get ParserManager and waiting for parser
		ParserManager pm = ParserManager.get(document);
		try
		{
			node = pm.getAST();
			//while (counter < 20 && pm.getState() == ParserManager.State.PARSING)
			while (counter < 20 && !node.getChildren().iterator().hasNext())
			//while (counter < 100 && !node.getChildren().iterator().hasNext())
			{
				try { Thread.sleep(100); } catch (InterruptedException ex) { }
				counter++;
				node = pm.getAST();
			}
			//node = pm.getAST();
		}
		catch (ParseException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		//boolean iserr = pm.hasSyntaxErrors();

		return node;
	}

	public static TokenSequence getTokenSequence (Document document)
	{
		TokenSequence ts = TokenHierarchy.get(document).tokenSequence();
		return ts;
	}

	public static Token previousToken (TokenSequence ts, boolean skipWs)
	{
		do
		{
			if (!ts.movePrevious())
				return null;
		}
		while (skipWs && (ts.token().id().name().endsWith("whitespace") ||
				ts.token().id().name().endsWith("comment")));

		return ts.token();
	}

	public static Token nextToken (TokenSequence ts, boolean skipWs)
	{
		do
		{
			if (!ts.moveNext())
				return null;
		}
		while (skipWs && (ts.token().id().name().endsWith("whitespace") ||
				ts.token().id().name().endsWith("comment")));

		return ts.token();
	}

	// -----------------------------------------------------------------

	// read types and methods declarations from file
	// if importOnly == true file parsed until first occurrence of ':- implementation '
	// XXX TODO importOnly doesn't support several modules in one file
	public static List<MercuryCompletionItem> getFileMethods (FileObject file,
																	boolean importOnly)
	{
		List<MercuryCompletionItem> list = new ArrayList<MercuryCompletionItem>();

		// open source file
		BufferedReader br = null;
		try
		{
			InputStream is = file.getInputStream();
			InputStreamReader isr = new InputStreamReader(is);
			br = new BufferedReader(isr);
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		if (br == null)
			return list;

		// parse file

		// XXX TODO fix support for comments parsing
		// XXX TODO use StringBuilder
		try
		{
			String line = "";
			String name, decl;
			String library = "";
			String comment = "";
			boolean is_import = false;
			boolean is_comment = false;
			int linesz = 0;
			int offset = 0;

			while ((offset += linesz) >= 0 &&
					(line = br.readLine()) != null)
			{
				linesz = line.getBytes().length + 1; // XXX TODO new line size in windows
				line = line.trim();

				// XXX TODO simple check for comments, fix this

				if (is_comment || line.startsWith("/*") || line.startsWith("%"))
				{
					if (line.startsWith("/*") && line.endsWith("*/"))
					{
						line = line.substring(2, line.length() - 2);
					}
					else if (line.endsWith("*/"))
					{
						line = line.substring(0, line.length() - 2);
						is_comment = false;
					}
					else if (is_comment)
					{}
					else if (line.startsWith("/*"))
					{
						line = line.substring(2);
						is_comment = true;
					}
					else
					{
						// % comment
						line = line.substring(1);
					}

					if (!comment.isEmpty())
						comment += "\n";
					comment += line.trim();

					continue;
				} // if (is_comment

				// check for declaration

				line = cleanLine(line);

				if (!line.startsWith(":-") ||
					(line.indexOf(" module") == -1 &&
					 line.indexOf(" interface") != -1 && line.indexOf(" implementation") != -1 &&
					 line.indexOf(" pred") != -1      && line.indexOf(" func") != -1 &&
					 line.indexOf(" type") != -1))
				{
					if (!"".equals(comment)) comment = "";
					continue;
				}

				// module declaration
				if (!"".equals(name = getModuleName(line, true)))
				{
					library = name;
					if (!"".equals(comment)) comment = "";
					continue;
				}
				// import
				else if (line.indexOf(" interface ") != -1 || line.endsWith(" interface."))
				{
					is_import = true;
					if (!"".equals(comment)) comment = "";
					continue;
				}
				// implementation
				else if (line.indexOf(" implementation ") != -1 || line.endsWith(" implementation."))
				{
					is_import = false;
					if (importOnly)
						break; // interface end, exit
				}

				// type, func or pred declaration

				decl = line;
				while (!decl.endsWith("."))
				{
					// read multiline decl
					if ((line = br.readLine()) == null)
						break;
					linesz += line.getBytes().length + 1;
					decl += " " + cleanLine(line);
				}

				if ("".equals(name = getMethodName(decl, true)) &&
						"".equals(name = getTypeName(decl, true)))
				{
					if (!"".equals(comment)) comment = "";
					continue;
				}

				// for debug
//				String sDbg = name;
//				if (sDbg.indexOf("dump_var(var(T)::ia, io::di, io::uo)") != -1)
//					sDbg += "";

				if (importOnly && !is_import)
					continue; // skip all util "interface" declaration (that may be?)

				// ok, declaration
				// also, save node filename and offset to description

				addCompletionInfo(list, name, "", library,
									CompletionItem.Type.LOCAL, 200,
									file.getPath(), offset, 0);

				if (!"".equals(comment)) comment = "";
			} // while

			br.close();
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		return list;
	}

	// add full completion info to list or map
	public static boolean addCompletionInfo (Object list, String text,
													String description, String library,
													CompletionItem.Type type, int priority,
													String filePath, int declOffset, int srcOffset)
	{
		boolean isMap = false;

		if (list instanceof List) {}
		else if (list instanceof Map) isMap = true;
		else return false;

//		if (text.indexOf("write_string") != -1)
//			isMap = isMap;

		if (library != null && !library.isEmpty() && text.startsWith(library + "."))
			// remove library prefix from method name
			try { text = text.substring((library + ".").length()); } catch (Exception ex) { }

		//
		MercuryCompletionItem item =
			MercuryCompletionItem.create(text, description, library, type, priority,
											filePath, declOffset, srcOffset);
		if (isMap)
			((Map<String, MercuryCompletionItem>) list).put(text, item);
		else
			((List<MercuryCompletionItem>) list).add(item);

		// add without params
		// XXX TODO check with types
		String sShortName = getMethodShortName(text);
		if (!"".equals(sShortName))
		{
			item = item.cloneItem().setText(sShortName);

			if (isMap)
				((Map<String, MercuryCompletionItem>) list).put(sShortName, item);
			else
				((List<MercuryCompletionItem>) list).add(item);
		}

		// get new name with '!IO'
		String sIoName = fixMethodNameIO(text);
		if (!"".equals(sIoName))
		{
			item = item.cloneItem().setText(sIoName);

			if (isMap)
				((Map<String, MercuryCompletionItem>) list).put(sIoName, item);
			else
				((List<MercuryCompletionItem>) list).add(item);
		}

		return true;
	}
}
