
/**
 *
 * @author gloom@opcode.cc 2017
 */

package org.mercury.lang;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.swing.text.Document;
import javax.swing.text.AbstractDocument;
import org.mercury.global.MercuryInterface;
import org.mercury.gui.MainOptionsPanel;
import org.netbeans.api.languages.ASTNode;
import org.netbeans.api.languages.ASTPath;
import org.netbeans.api.languages.ASTToken;
import org.netbeans.api.languages.CompletionItem;
import org.netbeans.api.languages.Context;
import org.netbeans.api.languages.LibrarySupport;
import org.netbeans.api.languages.SyntaxContext;
import org.netbeans.api.lexer.Token;
import org.netbeans.api.lexer.TokenHierarchy;
import org.netbeans.api.lexer.TokenSequence;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.DataObject;
import org.openide.windows.OutputWriter;

public class NBS extends ParserHelper
{
	// predefined mercury library
	private static final String   mercuryLibKeywords = "org/mercury/lang/language.xml";
	private static final String   mercuryLibTest     = "org/mercury/lang/library.xml"; // for debug
	private static LibrarySupport mercuryStaticLibrary = null;

	// mercury library from installation
	private static List<MercuryCompletionItem> mercuryLibrary = null;
	private static final Object mercuryLibraryLock = new Object();

	// mercury project files cache
	private static final Map<String, MercuryFileCompletionCache> mercuryCacheLibrary =
		new HashMap<String, MercuryFileCompletionCache>();

	// mercury.nbs definitions
	private static final String typeImage          = "org/mercury/v2type.png";
	private static final String methodDeclareImage = "org/mercury/v2method_decl.png";
	private static final String methodImage        = "org/mercury/v2method.png";
	//private static final String typeImage          = "org/mercury/vstype_decl.png";
	//private static final String methodDeclareImage = "org/mercury/vsmethod_decl.png";
	//private static final String methodImage        = "org/mercury/vsmethod.png";

	private static final String statementEndNode = "SEnd";
	private static final String declarationNode  = "Decls";
	private static final String foreignProcNode  = "FProc";
	private static final String methodNode0      = "Function";
	private static final String methodNode1      = "SHead";

	// -----------------------------------------------------------------

	public static void initLibrary ()
	{
		if (mercuryStaticLibrary != null)
			return;

		synchronized (mercuryLibraryLock)
		{
			if (mercuryStaticLibrary != null) // XXX TODO fix this
				return;

			mercuryStaticLibrary =
				LibrarySupport.create(Arrays.asList(
//					new String[] {mercuryLibKeywords, mercuryLibTest})); // for debug
					new String[] {mercuryLibKeywords}));

			InitCompilerLibrary();

			mercuryLibrary = new ArrayList<MercuryCompletionItem>();
			LoadCompilerLibrary();
		}
	}

	private static void InitCompilerLibrary ()
	{
	}

	// code completion, load declarations from mercury '.int' files
	// see completionItems
	private static void LoadCompilerLibrary ()
	{
		OutputWriter ow = MercuryInterface.getOutputWindow();
		ow.println();

		// first see if we have a compiler available
		String mercuryPath = MainOptionsPanel.getMercuryPath();
		if (mercuryPath.isEmpty())
		{
			ow.println("You need to set up Mercury install path for code completion and restart IDE.");
			ow.println("In the main menu go to 'Tools / Options / Miscellaneous / Mercury'.");
			ow.println();
			ow.println();

			ow.close();
			return;
		}

		ow.println("Parsing mercury libs, please wait...");

		// parse all '.int' files from '$mercuryLibPath\ints'
		File lib_path = new File(mercuryPath + MainOptionsPanel.MERCURY_INTS_SUBDIR);
		FileObject dir = null;
		if (lib_path.isDirectory())
			dir = FileUtil.toFileObject(lib_path);

		Enumeration files = (dir == null) ? null : dir.getChildren(false);
		int count = 0;
		while (files != null && files.hasMoreElements())
		{
			FileObject file = (FileObject) files.nextElement();
			if (file.isData() && file.hasExt("int"))
			{
				// found '.int' file, parse
				List<MercuryCompletionItem> items = getFileMethods(file, true);
				//if (items != null)
				{
					mercuryLibrary.addAll(items);
					count++;
				}
			}
		}

		ow.println(count + " files parsed");
		ow.println();
		ow.println();

		ow.close();
	}

	// project files cache (see getImportMethods)
	private static class MercuryFileCompletionCache
	{
		File file;
		List<MercuryCompletionItem> items;
		long modify;

		MercuryFileCompletionCache (String filePath, List<MercuryCompletionItem> items)
		{
			this.items = items;
			this.file = new File(filePath);

			if (this.file.isFile())
				this.modify = this.file.lastModified();
		}

		public List<MercuryCompletionItem> getItems ()
		{
			return this.items;
		}

		public boolean isValid()
		{
			if (this.modify == 0)
				return false;

			if (this.file.isFile())
			{
				long modify0 = this.file.lastModified();
				if (this.modify == modify0)
					return true;
			}

			return false;
		}
	}

	private static List<MercuryCompletionItem> getFromCacheLibrary (String filePath)
	{
		synchronized (mercuryLibraryLock)
		{
			MercuryFileCompletionCache c = mercuryCacheLibrary.get(filePath);
			if (c != null && !c.isValid())
			{
				mercuryCacheLibrary.remove(filePath);
				c = null;
			}

			return ((c != null) ? c.getItems() : null);
		}
	}

	private static void updateCacheLibrary (String filePath, List<MercuryCompletionItem> items)
	{
		synchronized (mercuryLibraryLock)
		{
			MercuryFileCompletionCache c = new MercuryFileCompletionCache(filePath, items);
			if (c.isValid())
				mercuryCacheLibrary.put(filePath, c);
		}
	}

	// -----------------------------------------------------------------

	// return true if node contains type
	private static boolean isTypeNode (ASTNode node0)
	{
		ASTNode node = node0.getNode(declarationNode);
		if (node == null || !"type".equals(node.getAsText().trim()))
			return false;

		return true;
	}

	// return true if node contains foreign proc implementation
	private static boolean isForeignProcNode (ASTNode node0)
	{
		ASTNode node = node0.getNode(foreignProcNode);
		return (node != null);
	}

	// return true if node contains func or pred implementation
	private static boolean isMethodNode (ASTNode node0)
	{
		ASTNode node = node0.getNode(methodNode0);
		if (node == null)
			node = node0.getNode(methodNode1);

		if (node != null)
			return true;

		return false;
	}

	// return node name if node contains:
	//     type declaration
	//     func, pred implementation or declaration
	//     foreign proc implementation
	// return "" if unknown node
	private static String getNodeName (ASTNode node0)
	{
		String sName;

		// check end of statement
		ASTNode nodeE = node0.getNode(statementEndNode);
		if (nodeE == null || nodeE.getLength() == 0)
			return "";

		// check statement type

		ASTNode node;
		if ((node = node0.getNode(declarationNode)) != null &&
				"type".equals(node.getAsText().trim()))
		{
			// type (see isTypeNode)
			sName = getTypeName(node0.getAsText());
		}
		else if ((node = node0.getNode(foreignProcNode)) != null)
		{
			// foreign proc (see isForeignProcNode)
			sName = getForeignProcName(node0.getAsText());
		}
		else
		{
			// check for func or pred (see isMethodNode)
			node = node0.getNode(methodNode0);
			if (node == null)
				node = node0.getNode(methodNode1);

			if (node != null)
				// func or pred implementation node, get text
				sName = cleanLine(node.getAsText());
			else
				// func or pred declaration, may be
				sName = getMethodName(node0.getAsText());
		}

		//return language_token.getIdentifier().split(":")[1].trim();
		return sName;
	}

	// -----------------------------------------------------------------

	private static ASTNode getActualNode (SyntaxContext context)
	{
		ASTPath path = context.getASTPath();
		ASTNode node = (ASTNode) path.getLeaf();

		return node;
	}

	private static ASTToken getActualToken (SyntaxContext context)
	{
		ASTPath path = context.getASTPath();
		ASTToken token = (ASTToken) path.getLeaf();

		return token;
	}

	// return node text for navigator
	// if this function return != "" then next NBS call - statementIcon func
	public static String statementName (SyntaxContext context)
	{
		ASTNode node = getActualNode(context);

		// for debug
//		String sDbg = node.getAsText();
//		ASTNode nodeT = node.getNode("ListOfStructures");
//		if (sDbg.indexOf("C = (pred(A::in, B::out)") != -1)
//		if (sDbg.indexOf("type person") != -1)
//		if (sDbg.indexOf("foreign_proc") != -1)
//			sDbg += "";

		String sName = getNodeName(node);
		return sName;
	}

	// return node icon for navigator
	public static String statementIcon (SyntaxContext context)
	{
		ASTNode node = getActualNode(context);

		if (isTypeNode(node))
			return typeImage;

		if (isForeignProcNode(node) || isMethodNode(node))
			return methodImage;

		return methodDeclareImage;
	}

	/*
	// on true next call - methodName func
	public static boolean isMethodDeclaration (SyntaxContext context)
	{
		ASTToken token = getActualToken(context);
		int p = token.getOffset();
		String s = token.getIdentifier();
		if (p == 183 && s.equals("main2"))
			return true;

		return false;
	}

	public static boolean isMethodUsage (SyntaxContext context)
	{
		ASTToken token = getActualToken(context);
		int p = token.getOffset();
		String s = token.getIdentifier();
		if (p != 183 && s.equals("main2"))
			return true;

		return false;
	}

	//
	public static String methodName (SyntaxContext context)
	{
		//String mName = statementName(context);
		//return mName;
		//return "";
		return "main2";
	}
	*/

	// code completion, fill completion list for current token
	public static List<MercuryCompletionItem> completionItems (Context context)
	{
		if (context instanceof SyntaxContext)
			return (new ArrayList<MercuryCompletionItem>());

		//TokenSequence ts = context.getTokenSequence();
		Document document = context.getDocument();
		int offset = context.getOffset();

		return completionItems(document, offset, false);
	}

	// -----------------------------------------------------------------

	public static List<MercuryCompletionItem> completionItems (Document document, int offset,
																	boolean getAll)
	{
		List<MercuryCompletionItem> items = new ArrayList<MercuryCompletionItem>();
		String lib = null;

		// init completion library (parse mercury libs info and other)
		initLibrary();

		// parse current module and imported modules

		((AbstractDocument) document).readLock();
		try
		{
			// get current token
			TokenSequence ts = getTokenSequence(document);
			if (ts == null) // hmm, when?
				return (new ArrayList<MercuryCompletionItem>());

			ts.move(offset);
			Token token = previousToken(ts, true);

			if (token == null) // hmm, when?
				return (new ArrayList<MercuryCompletionItem>());

			String tokenText = token.text().toString();
			String tokenName = token.id().name();

			if (!getAll && tokenText.length() == 1 && tokenText.equals("."))
			{
				// may be 'lib.' completion? get prev text
				// XXX TODO not work with 'module1.subname.funcname'

				Token tokenLib = previousToken(ts, false);
				if (tokenLib != null)
				{
					String tokenLibText = tokenLib.text().toString().trim();
					if (!tokenLibText.isEmpty())
						lib = tokenLibText;
				}
			}

			// check token type
			if ("field".equals(tokenName)) // variable, see mercury.nbs comment
			{
				items.addAll(getMethodVars(document, offset));
			}
			else //if ("identifier".equals(tokenName))
			{
				items.addAll(MercuryCompletionItem.fromList(
								mercuryStaticLibrary.getCompletionItems("root")));
				items.addAll(mercuryLibrary);

				items.addAll(getDocumentMethods(document, null));
				items.addAll(getImportMethods(document));
			}
		}
		finally
		{
			((AbstractDocument) document).readUnlock();
		}

		return mergeItems(items, lib);
	}

	// merge MercuryCompletionItems from Library (add 'lib.' and 'lib__' prefixes)
	private static List<MercuryCompletionItem> mergeItems (List<MercuryCompletionItem> items,
																String lib)
	{
		Map<String, MercuryCompletionItem> map = new HashMap<String, MercuryCompletionItem>();
		Iterator<MercuryCompletionItem> it = items.iterator();

		while (it.hasNext())
		{
			MercuryCompletionItem complItem = it.next();
			MercuryCompletionItem complItemE;

//			if (complItem.getText().indexOf("write_string") != -1)
//				lib += "";

			if (lib != null)
			{
				if (!lib.equals(complItem.getLibrary()))
					continue;

				// XXX TODO '.' completion workaround, add funcs from selected lib with '.' prefix

				String lib_name = "." + complItem.getText();
				complItemE = complItem.cloneItem().setText(lib_name);
				map.put(lib_name, complItemE);

				continue;
			}

			// add normal item (write_string)

			MercuryCompletionItem item = map.get(complItem.getText());
			if (item == null)
			{
				// new method name
				map.put(complItem.getText(), complItem);
			}
			else
			{
				// found method with same name in other lib

				String library = item.getLibrary();
				if (library == null)
					library = "";

				String library_new = complItem.getLibrary();
				if (library_new == null)
					library_new = "";

				// XXX TODO fix this
				String library_s = " " + library + " ";

				if ("".equals(library_new)) {}
				else if ("".equals(library))
					library = library_new;
				else if (library_s.indexOf(" " + library_new + " ") != -1) {}
				else
					library += " " + library_new;

				//
				complItemE = complItem.cloneItem().setLibrary(library);
				map.put(complItemE.getText(), complItemE);
			}

			// add library prefix (io.write_string, io__write_string)
			// XXX TODO move it to addCompletionInfo

			if (complItem.getType() != CompletionItem.Type.LOCAL ||
					"".equals(complItem.getLibrary()))
			{
				// don't add library prefix for keywords, vars and methods without lib
				continue;
			}

			String lib_name = complItem.getLibrary().replaceAll("__", ".") +
								"." + complItem.getText();

			complItemE = complItem.cloneItem().setText(lib_name);
			map.put(lib_name, complItemE);

			lib_name = complItem.getLibrary().replaceAll("\\.", "__") +
								"__" + complItem.getText();

			complItemE = complItem.cloneItem().setText(lib_name);
			map.put(lib_name, complItemE);
		} // while

		return (new ArrayList<MercuryCompletionItem>(map.values()));
	}

	// get types and methods list from document
	private static List<MercuryCompletionItem> getDocumentMethods (Document document,
																		FileObject file)
	{
		Map<String, MercuryCompletionItem> map = new HashMap<String, MercuryCompletionItem>();

		ASTNode docAST = getDocumentAST(document);
		if (docAST == null)
			return (new ArrayList<MercuryCompletionItem>(map.values()));

		// get methods declarations from document

//		String sDbg = docAST.getAsText();
		Iterator it = docAST.getChildren().iterator();
		String library = "";

		while (it.hasNext())
		{
			Object elem = it.next();
			if (!(elem instanceof ASTNode))
				continue;

			// check node params
			String sNodeName = getNodeName((ASTNode) elem);
			int offset = ((ASTNode) elem).getOffset();
			MercuryCompletionItem item;

			if ("".equals(sNodeName))
			{
				// not type, func or pred, check for node with module name
				String libraryN = getModuleName(((ASTNode) elem).getAsText());
				if (!"".equals(libraryN))
					library = libraryN;

				continue;
			}
			else
			{
				final String sShortName = getMethodShortName(sNodeName);
				final String sName = !sShortName.isEmpty() ? sShortName : sNodeName;
				item = map.get(sName); // see addCompletionInfo

//				if (sNodeName.indexOf("var.rep_free") != -1)
//					sNodeName += "";

				if (item != null)
				{
					// node already added (short name found)

					// XXX TODO isTypeNode doesn't check if type decl has body (has == or --->)
					boolean isSource =
							(isTypeNode((ASTNode) elem) || isForeignProcNode((ASTNode) elem) ||
								isMethodNode((ASTNode) elem));

					if (isSource && item.getSrcOffset() == 0)
						//!"".equals(sShortName)
					{
						// node with type, func or pred body
						// update method src info, set to first source position in file

						item.setSrcOffset(offset);
						map.put(sName, item);
					}

					continue;
				}
			}

			// ok, declaration, node has name and node not func or pred body

			String filePath = (file == null) ? null : file.getPath();
			addCompletionInfo(map, sNodeName, "", library,
								CompletionItem.Type.LOCAL, 200,
								filePath, offset, 0);
		} // while

		return (new ArrayList<MercuryCompletionItem>(map.values()));
	}

	// get types and methods list from imported modules
	private static List<MercuryCompletionItem> getImportMethods (Document document)
	{
		List<MercuryCompletionItem> list = new ArrayList<MercuryCompletionItem>();
		Map<String, String> modules = null;

		ASTNode docAST = getDocumentAST(document);
		if (docAST == null)
			return list;

		// search import names in document

//		String sDbg = docAST.getAsText();
		Iterator it = docAST.getChildren().iterator();

		while (it.hasNext())
		{
			Object elem = it.next();
			if (!(elem instanceof ASTNode))
				continue;

			// check if import
			String[] mod_import = getImportNames(((ASTNode) elem).getAsText());
			if (mod_import.length == 0)
				continue;

			// ok, found import

			// get info about all modules in project
			if (modules == null)
			{
				// XXX TODO getModules not work with MercuryInterface.createDocument docs
				modules = MercuryInterface.getModules(document);
				if (modules == null || modules.isEmpty())
					break;
			}

			// parse imported modules import
			List<MercuryCompletionItem> items;
			for (int i = 0; i < mod_import.length; i++)
			{
				String path = modules.get(mod_import[i].trim());
				if (path == null)
					continue;

				items = getFromCacheLibrary(path);
				if (items == null)
				{
					DataObject data_obj = MercuryInterface.getFile(path);
					if (data_obj == null)
						continue;
					FileObject file = data_obj.getPrimaryFile();
					if (file == null || !file.isData())
						continue;

					//items = getFileMethods(file, true);
					Document doc = MercuryInterface.createDocument(file);
					if (doc == null)
						continue;
					items = getDocumentMethods(doc, file);

					if (items != null)
						updateCacheLibrary(path, items);
				}

				//if (items != null)
					list.addAll(items);
			}
		} // while

		return list;
	}

	// get vars list from document
	// search variables from current pos up and down until '.' not found
	private static List<MercuryCompletionItem> getMethodVars (Document document, int docCurPos)
	{
		Map<String, MercuryCompletionItem> map = new HashMap<String, MercuryCompletionItem>();

		TokenHierarchy th = TokenHierarchy.get(document);
		TokenSequence ts = th.tokenSequence();
		Token token;
		String tokenText;
		int tokenPos;

		// get pos of prev and next tokens
		ts.move(docCurPos);
		token = previousToken(ts, true);
		int prevTokenPos = (token == null) ? 0 : token.offset(th);
		ts.move(docCurPos);
		token = nextToken(ts, true);
		int nextTokenPos = (token == null) ? 0 : token.offset(th);

		// search back, previous '.'
		ts.move(docCurPos);
		while ((token = previousToken(ts, true)) != null &&
				!".".equals(token.text().toString().trim()))
		{}

		// search forvard, next '.'
		while ((token = nextToken(ts, true)) != null)
		{
			if (".".equals((tokenText = token.text().toString().trim())))
				break;
			else if (!"field".equals(token.id().name())) // variable, see mercury.nbs comment
				continue;

			// check if found current token
			// if add current incomplete token to list, autocomplete may 'eat' some chars
			tokenPos = token.offset(th);
			if (tokenPos >= prevTokenPos && tokenPos <= nextTokenPos)
				continue;

			// found variable
			if (map.get(tokenText) != null)
				continue; // already added, skip

			// add variable
			MercuryCompletionItem item =
				MercuryCompletionItem.create(tokenText, "", "",
										CompletionItem.Type.PARAMETER, 200);
			map.put(tokenText, item);

			// add variable name with !
			if (!tokenText.startsWith("!"))
			{
				item = item.cloneItem().setText("!" + tokenText);
				map.put(item.getText(), item);
			}
		}

		return (new ArrayList<MercuryCompletionItem>(map.values()));
	}
}
