
/**
 *
 * @author gloom@opcode.cc 2017
 */

package org.mercury.compiler;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.text.StyledDocument;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.mercury.lang.ParserHelper;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.openide.awt.StatusDisplayer;
import org.openide.cookies.EditorCookie;
import org.openide.cookies.LineCookie;
import org.openide.cookies.SaveCookie;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.DataObject;
import org.openide.nodes.Node;
import org.openide.text.Line;
import org.openide.text.NbDocument;
import org.openide.util.Exceptions;
import org.openide.util.Utilities;

public class CompilerHelper implements Runnable
{
	// project dirs
	public static final String PROJECT_BUILDDIR = "build";
	public static final String PROJECT_DIR      = "mercury";
	public static final String PROJECT_SRCDIR   = "source";

	// project build files
	public static final String PROJECT_MODSFILE = "Mercury.modules";
	public static final String PROJECT_OPTSFILE = "Mercury.options";
	public static final String PROJECT_MAKEFILE = "Mmakefile";
	public static final String PROJECT_FILE     = "Mercury.project"; // not mercury file

	// project config
	public static final String CONFIG_MAIN_SOURCE   = "main_file";

	public static final String CONFIG_REMOTE_BUILD  = "remote_build";  // -> "true" or "false"
	public static final String CONFIG_REMOTE_LOGIN  = "remote_login";
	public static final String CONFIG_SSH_CMD       = "ssh_command";
	public static final String CONFIG_RSYNC_CMD     = "rsync_command";
	public static final String CONFIG_REMOTE_PATH   = "remote_path";   // remote path to upload files (project root dir and other)
	public static final String CONFIG_REMOTE_UPLOAD = "remote_upload"; // -> remote_upload0:path1, remote_upload1:path2
	public static final String CONFIG_REMOTE_BINS   = "remote_bins";

	protected Map<String, String> config = new HashMap<String, String>();

	// used for compiling file
	protected DataObject dataObject = null;
	protected FileObject dataObjectFile = null;

	// used for making project
	protected Project projectObject = null;

	// http://www.regexplanet.com/advanced/java/index.html
	protected static final Pattern pattern_module =
						Pattern.compile("[^%](:-)[\\s]+(module)[\\s]+(\\([\\w.()]+\\)|[\\w]+)[\\s]*(\\.)");

	// -----------------------------------------------------------------

	// initial settings for compiling file or make project
	public CompilerHelper (DataObject dObj)
	{
		setDataObject(dObj);
		this.config = projectGetConfig();
	}

	// initial settings for make project
	public CompilerHelper (Project pObj)
	{
		setProjectObject(pObj);
		this.config = projectGetConfig();
	}

	// initial settings to call helper functions
	public CompilerHelper ()
	{
	}

	public void run ()
	{
		throw new UnsupportedOperationException("Not supported yet.");
	}

	protected final void setDataObject (DataObject dObj)
	{
		if (dObj != null)
		{
			// set DataObject and file name if compile single file
			this.dataObjectFile = dObj.getPrimaryFile();
//			if (this.dataObjectFile == null)
//				throw new IllegalArgumentException();
			if (this.dataObjectFile != null)
				this.dataObject = dObj;
		}
		else
		{
//			throw new IllegalArgumentException();
		}
	}

	protected final void setProjectObject (Project pObj)
	{
		if (pObj != null)
		{
			// set Project if making project
			this.projectObject = pObj;
		}
		else
		{
//			throw new IllegalArgumentException();
		}
	}

	// -----------------------------------------------------------------

	// getting project root directory
	public final FileObject projectGetRootDirectory ()
	{
		Project project;
		if (this.dataObject != null)
			project = FileOwnerQuery.getOwner(this.dataObject.getPrimaryFile());
		else
			project = this.projectObject;

		if (project == null)
			return null;

		FileObject rootDir = project.getProjectDirectory();
		return rootDir;
	}

	// getting project subdirectory with mercury options and build files
	public final FileObject projectGetProjectDirectory ()
	{
		FileObject rootDir = projectGetRootDirectory();
		if (rootDir == null)
			return null;

		// check project dir
		// XXX TODO check project file
		FileObject project_dir = rootDir.getFileObject(CompilerHelper.PROJECT_DIR);
		if (project_dir != null && project_dir.isFolder())
			return project_dir;

		return null;
	}

	// getting project subdirectory with mercury source files
	public final FileObject projectGetSourceDirectory ()
	{
		FileObject rootDir = projectGetRootDirectory();
		if (rootDir == null)
			return null;

		// check source dir
		FileObject source_dir = rootDir.getFileObject(CompilerHelper.PROJECT_SRCDIR);
		if (source_dir != null && source_dir.isFolder())
			return source_dir;

		return null;
	}

	// get project subdirectory for build files
	public FileObject projectGetBuildDirectory ()
	{
		FileObject rootDir = projectGetRootDirectory();
		if (rootDir == null)
			return null;

		//outputWriter.printf("Project directory is %s\n\n", rootDir.getPath());
		return createBuildDirectory(rootDir);
	}

	//
	public FileObject createBuildDirectory (FileObject rootDir)
	{
		FileObject buildDir = null;
		try
		{
			buildDir = rootDir.getFileObject(CompilerHelper.PROJECT_BUILDDIR);
			if (buildDir == null)
				buildDir = rootDir.createFolder(CompilerHelper.PROJECT_BUILDDIR);

			//buildDir_f = FileUtil.toFile(buildDir);
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		return buildDir;
	}

	//
	public void cleanBuildDirectory (FileObject dirObj, boolean deleteAll)
	{
		//OutputWriter ow = MercuryInterface.getOutputWindow("Mercury debug", true);

		dirObj.refresh(false);
		Enumeration files = dirObj.getChildren(false);
		//ow.println("dirObj: " + dirObj.getNameExt() + " " + dirObj.getPath());

		while (files.hasMoreElements())
		{
			FileObject file = (FileObject) files.nextElement();
			//ow.println("file: " + file.getNameExt() + " " + file.getPath());

			if (deleteAll ||
				(file.isData() && !file.hasExt("err") && !file.hasExt("mh")))
			{
				try { file.delete(); } catch (IOException ex) { Exceptions.printStackTrace(ex); }
				//ow.println("del");
			}
		}
	}

	// see Mercury.project and CONFIG_*
	public final Map<String, String> projectGetConfig ()
	{
		Map<String, String> config0 = new HashMap<String, String>();
		FileObject projectDir = projectGetProjectDirectory();

		// open file Mercury.project in mercury dir
		BufferedReader br = null;
		try
		{
			if (projectDir != null)
			{
				FileObject project_file = projectDir.getFileObject(CompilerHelper.PROJECT_FILE);
				if (project_file != null && project_file.isData())
				{
					InputStream is = project_file.getInputStream();
					InputStreamReader isr = new InputStreamReader(is);
					br = new BufferedReader(isr);
				}
			}
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		// parse Mercury.project file (JSON with our comments: line started with #)
		// XXX TODO rewrite with StringBuilder

		String sConfig = "";
		try
		{
			if (br != null)
			{
				String sLine;
				while ((sLine = br.readLine()) != null)
				{
					sLine = sLine.trim();
					if (sLine.isEmpty() || sLine.startsWith("#"))
						continue;

					sConfig += sLine;
				}

				br.close();
			}
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		if (sConfig.isEmpty())
			sConfig = "{ }";

		// XXX TODO rewrite this
		// some data may be missing in config0 if exception occurs (see Compiler.InitSettings)
		try
		{
			JSONObject jConfig = new JSONObject(sConfig);

			config0.put(CompilerHelper.CONFIG_MAIN_SOURCE, jConfig.optString(CompilerHelper.CONFIG_MAIN_SOURCE).trim());

			if (jConfig.has(CompilerHelper.CONFIG_REMOTE_BUILD))
				config0.put(CompilerHelper.CONFIG_REMOTE_BUILD, (jConfig.getBoolean(CompilerHelper.CONFIG_REMOTE_BUILD) ? "true" : "false"));
			else
				config0.put(CompilerHelper.CONFIG_REMOTE_BUILD, "false");

			config0.put(CompilerHelper.CONFIG_REMOTE_LOGIN, jConfig.optString(CompilerHelper.CONFIG_REMOTE_LOGIN).trim());
			config0.put(CompilerHelper.CONFIG_RSYNC_CMD,    jConfig.optString(CompilerHelper.CONFIG_RSYNC_CMD).trim());
			config0.put(CompilerHelper.CONFIG_SSH_CMD,      jConfig.optString(CompilerHelper.CONFIG_SSH_CMD).trim());

			config0.put(CompilerHelper.CONFIG_REMOTE_PATH,  jConfig.optString(CompilerHelper.CONFIG_REMOTE_PATH).trim());
			if (jConfig.has(CompilerHelper.CONFIG_REMOTE_UPLOAD))
			{
				JSONArray a = jConfig.getJSONArray(CompilerHelper.CONFIG_REMOTE_UPLOAD);
				int i, c = a.length();
				for (i = 0; i < c; i++)
					// TODO XXX skip empty
					config0.put(CompilerHelper.CONFIG_REMOTE_UPLOAD + i, a.getString(i).trim());
			}

			config0.put(CompilerHelper.CONFIG_REMOTE_BINS,  jConfig.optString(CompilerHelper.CONFIG_REMOTE_BINS).trim());
		}
		catch (JSONException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		return config0;
	}

	// get mercury modules list in current project
	// sample result - see filesGetModules
	public Map<String, String> getModules ()
	{
		FileObject projectDir = projectGetProjectDirectory();
		FileObject projectSrcDir = projectGetSourceDirectory();

		return projectGetModules(projectDir, projectSrcDir);
	}

	// tries to find file in current project
	// filename may be file name without path
	public DataObject getFile (String filename)
	{
		DataObject dObj = null;

		try
		{
			// XXX TODO fix DataObject search
			FileObject rootDir = projectGetRootDirectory();
			File s_file = new File(filename);
			FileObject fObj = null;

			if (s_file.isFile())
			{
				fObj = FileUtil.toFileObject(s_file);
			}
			else if (rootDir != null)
			{
				// file without full path, try find it throw all project files

				// FileObject getPath always use / as separator and return path without splash at end !!!
				String search_name = "/" + filename;
				Enumeration files = rootDir.getChildren(true);

				String sBuildDir = null;
				FileObject buildDir = projectGetBuildDirectory();
				if (buildDir != null)
					sBuildDir = buildDir.getPath() + "/";

				while (files.hasMoreElements())
				{
					FileObject file = (FileObject) files.nextElement();

//					if (file.getPath().indexOf("main.m") != -1) {
//						String t1 = file.getPath();
//						boolean t2 = file.isData();
//						search_name += "";
//					}

					if (file.isData() && file.getPath().endsWith(search_name) &&
						(sBuildDir == null || !file.getPath().startsWith(sBuildDir)))
					{
						fObj = file;
						break; // found
					}
				}
			}
			else
			{
				fObj = FileUtil.toFileObject(s_file);
			}

			if (fObj != null)
				dObj = DataObject.find(fObj);
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		return dObj;
	}

	//
	public static boolean saveFile (DataObject dObj)
	{
		SaveCookie sCookie = dObj.getLookup().lookup(SaveCookie.class);
		if (sCookie != null)
		{
			try
			{
				sCookie.save();
				StatusDisplayer.getDefault().setStatusText("Save finished");

				return true;
			}
			catch (IOException ex)
			{
				Exceptions.printStackTrace(ex);
			}
		}

		return false;
	}

	// -----------------------------------------------------------------

	// creates commmand string from array of params
	public static String commandToString (List<String> cmds)
	{
		return commandToString((String[]) cmds.toArray(new String[cmds.size()]));
	}

	public static String commandToString (String[] cmds)
	{
		StringBuilder sBuf = new StringBuilder();

		for (String cmd : cmds)
		{
			// XXX TODO "/s t/t1.sh" ; t2.sh not converted to "\"/s t/t1.sh\" ; t2.sh"
			// XXX TODO "/s t/t1.sh" ; "/s t/t2.sh" not converted to ""/s t/t1.sh" ; "/s t/t2.sh""
			sBuf.append((cmd.indexOf(' ') != -1 && (
								cmd.charAt(0) != '"' || cmd.charAt(cmd.length() - 1) != '"')) ?
							"\"" + cmd + "\"" : cmd);
			sBuf.append(' ');
		}
		// leave extra space on the end

		return sBuf.toString().trim();
	}

	// convert command or args string to list
	public static List<String> commandToArray (String cmd_str0)
	{
		//if (cmd_str0 == null)
		//	return null;

		String cmd_str = cmd_str0.trim();
		if (cmd_str.isEmpty())
			return null;

		List<String> sList = new ArrayList<String>();
		boolean ch_open = false;
		int last_pos = 0;

		for (int i = 0; i < cmd_str.length(); i++)
		{
			if (cmd_str.charAt(i) != '"')
				continue;

			if (!ch_open)
			{
				// " open
				String[] args = cmd_str.substring(last_pos, i).trim().split(" ");
				ch_open = true;
				last_pos = i;

				for (int j = 0; j < args.length; j++)
				{
					if (!args[j].trim().isEmpty())
						sList.add(args[j].trim());
				}
			}
			else
			{
				// " close
				sList.add(cmd_str.substring(last_pos, i + 1).trim());
				ch_open = false;
				last_pos = i + 1;
			}
		}

		String[] args = cmd_str.substring(last_pos, cmd_str.length()).trim().split(" ");
		for (int j = 0; j < args.length; j++)
		{
			if (!args[j].trim().isEmpty())
				sList.add(args[j].trim());
		}

		return sList;
	}

	// update process path env and return path string
	public static String processUpdatePathEnv (final ProcessBuilder procBuilder,
													String binsPath)
	{
		// get current env (can be modified)
		final Map<String, String> proc_env = procBuilder.environment();

		return processUpdatePathEnv(proc_env, binsPath);
	}

	public static  String processUpdatePathEnv (final Map<String, String> procEnv,
													String binsPath)
	{
		// search path var
		// XXX TODO fix path name detect
		String path_name = "PATH";
		if (procEnv.containsKey("PATH")) { }
		else if (procEnv.containsKey("Path")) path_name = "Path";
		else if (procEnv.containsKey("path")) path_name = "path";

		String path_val = procEnv.get(path_name);

		// modify proc env
		if (!binsPath.isEmpty())
		{
			path_val = (path_val == null) ? binsPath :
							binsPath + File.pathSeparator + path_val;
			procEnv.put(path_name, path_val);
		}

		return (path_val == null) ? "" : path_val;
	}

	// trying to get full path to binary using search in additional paths
	// if cmd under windows is 'sh mmc' than we need to correct it to 'sh.exe mmc'
	public static boolean processFixRunPath (final ProcessBuilder procBuilder,
													String runPath)
	{
		final List<String> proc_cmd = procBuilder.command();

		return processFixRunPath (proc_cmd, runPath);
	}

	public static boolean processFixRunPath (final List<String> procCmd,
													String runPath)
	{
		if (procCmd.isEmpty())
			return false;
		String prog = procCmd.get(0);

		File fileToRun = new File(prog);
		if (fileToRun.isFile())
			return false;

		// search file in runPath
		String[] dirs = runPath.split(File.pathSeparator);
		for (int i = 0; i < dirs.length; i++)
		{
			String path = dirs[i].trim();

			if (path.isEmpty())
				continue;

			// update command line for windows
			// XXX TODO also add ".bat" and test for it
			String prog0 = (Utilities.isWindows() &&
							!(prog.endsWith(".exe") || prog.endsWith(".bat"))) ?
					prog + ".exe" : prog;

			//
			path = pathAddSeparator(path) + prog0;

			fileToRun = new File(path);
			if (fileToRun.isFile())
			{
				procCmd.set(0, path);
				return true;
			}
		}

		return false;
	}

	public static String pathAddSeparator (String path)
	{
		//if (path.endsWith("\""))
		if (path.charAt(0) == '"')
			path = path.substring(1, path.length() - 1);

		// "/" or "\" can be used under win
		if (!path.endsWith("/") && !path.endsWith("\\"))
			path += File.separator;

		return path;
	}

	// -----------------------------------------------------------------

	// get modules list from source files and Mercury.modules file in project dir
	// sample result - see filesGetModules
	protected Map<String, String> projectGetModules (FileObject projectDir,
														FileObject projectSrcDir)
	{
		Map<String, String> modules = new HashMap<String, String>();

		// part of code from compileGenerateModulesList
		// XXX TODO remove duplicate code

		// check file Mercury.modules in mercury dir and open it
		BufferedReader br = null;
		try
		{
			if (projectDir != null)
			{
				FileObject modules_file = projectDir.getFileObject(CompilerHelper.PROJECT_MODSFILE);
				if (modules_file != null && modules_file.isData())
				{
					InputStream is = modules_file.getInputStream();
					InputStreamReader isr = new InputStreamReader(is);
					br = new BufferedReader(isr);
				}
			}
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		// parse Mercury.modules file
		try
		{
			if (br != null)
			{
				String sLine;
				while ((sLine = br.readLine()) != null)
				{
					String[] lines = sLine.trim().split("\t");
					if (lines.length < 2)
						continue;

					modules.put(lines[0].trim(), lines[1].trim());
				}

				br.close();
			}
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		// getting modules list from project source directory
		if (projectSrcDir != null)
			modules.putAll(filesGetModules(projectSrcDir));

		// getting modules list from current source directory
		if (this.dataObject != null)
		{
			FileObject source_dir = this.dataObjectFile.getParent();
			if (source_dir != null && !source_dir.equals(projectSrcDir))
				modules.putAll(filesGetModules(source_dir));
		}

		return modules;
	}

	// recursively getting modules list from start directory
	//
	// sample result:
	// main /root/test/source/main.m
	// test /root/test/source/test.m
	// test2 /root/test/source/main.m
	//
	// XXX TODO source files can be with another extensions?
	protected Map<String, String> filesGetModules (FileObject dir)
	{
		Map<String, String> modules = new HashMap<String, String>();

		// getting data files
		Enumeration files = dir.getChildren(true);
		while (files.hasMoreElements())
		{
			FileObject file = (FileObject) files.nextElement();
			if (file.isData() && file.hasExt("m"))
			{
				// getting modules info from file
				ArrayList<String> modules_f = sourceGetModulesInfo(file);
				if (modules_f.isEmpty())
				{
					// no module defenitions in file, use file name
					modules.put(file.getName(), file.getPath());
				}
				else
				{
					// find modules info
					Iterator modules_i = modules_f.iterator();
					while (modules_i.hasNext())
						modules.put((String)modules_i.next(), file.getPath());
				}
			} // if
		} // while

		return modules;
	}

	// getting modules names from mercury source file
	protected ArrayList<String> sourceGetModulesInfo (FileObject sourceFile)
	{
		ArrayList<String> modules = new ArrayList<String>();

		// open file and search module declaration (see getModuleName for sample)
		// in one file may be two or more modules
		try
		{
			String data = sourceFile.asText();

			Matcher matcher = CompilerHelper.pattern_module.matcher(" " + data + " ");
			while (matcher.find())
			{
				String module_str = data.substring(matcher.start(), matcher.end());
//				if (module_str.indexOf("(") != -1)
//					module_str += "";

				String name = ParserHelper.getModuleName(module_str);
				if (!"".equals(name))
					modules.add(name);
			}
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		return modules;
	}

	// -----------------------------------------------------------------

	// open file in editor at specified offset or line
	// filePath may be filename from current project without full path
	public boolean openEditor (String filePath, int offset, boolean isLineOffset,
									String statusText)
	{
		DataObject dObj = (filePath == null) ? this.dataObject : getFile(filePath);
		if (dObj != null)
		{
			LineCookie lCookie = dObj.getLookup().lookup(LineCookie.class);

			Line editorLine = null;
			try
			{
				if (isLineOffset)
				{
					editorLine = lCookie.getLineSet().getOriginal(offset);
				}
				else
				{
					Node dObjNode = dObj.getNodeDelegate();
					EditorCookie eCookie = dObjNode.getCookie(EditorCookie.class);
					StyledDocument doc = eCookie.openDocument();

					Line.Set lineSet = lCookie.getLineSet();
					int line = NbDocument.findLineNumber(doc, offset);
					editorLine = lineSet.getCurrent(line);
				}
			}
			catch (IndexOutOfBoundsException ex) { }
			catch (IOException ex) { }

			if (editorLine == null)
				return false;

			//editorLine.show(Line.SHOW_GOTO, 0); // columnNumber
			editorLine.show(Line.ShowOpenType.OPEN, Line.ShowVisibilityType.FOCUS, 0);

			if (statusText != null)
				StatusDisplayer.getDefault().setStatusText(statusText);

			return true;
		}

		return false;
	}

	/*
	public Map<String, String> getSystemEnv () throws Exception
	{
		Class[] classes = Collections.class.getDeclaredClasses();
		Map<String, String> env = System.getenv();
		for (Class cl : classes)
		{
			if ("java.util.Collections$UnmodifiableMap".equals(cl.getName()))
			{
				Field field = cl.getDeclaredField("m");
				field.setAccessible(true);
				Object obj = field.get(env);
				Map<String, String> map = (Map<String, String>) obj;
				//map.clear();
				//map.putAll(newenv);

				return map;
			}
		}

		return null;
	}
	*/
}
