
/**
 *
 * @author gloom@opcode.cc 2017
 */

package org.mercury.compiler;

import java.io.*;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import org.mercury.global.MercuryInterface;
import org.mercury.gui.MainOptionsPanel;
import org.netbeans.api.project.Project;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.Exceptions;
import org.openide.util.Utilities;
import org.openide.windows.*;


public class Compiler extends CompilerHelper
{
	private boolean useMmakeCmd; // use mmc or mmake command to compile/build file or project
	private String   dataObjectModule = null; // module name from mercury source file to compile

	// project settings
	private String mainSourceFile; // src file with main func (used to build project when mmake cmd is empty)

	// project remote build paths + settings
	private boolean      projectRemoteBuild;
	private String       projectRemoteLogin;
	private List<String> cmdSsh;
	private List<String> cmdRsync;
	private String       projectRemotePath;
	private final List<String> projectUploads = new ArrayList<String>();
	private String       projectRemoteBins; // remote PATH additionals

	// project paths
	private String projectRootPath;
	private String projectRootParentPath;

	// -----------------------------------------------------------------

	// see CompilerHelper class constructor
	public Compiler (DataObject dObj, boolean useMmakeCmd)
	{
		// set main objects
		super(dObj);
		if (this.dataObject == null)
			throw new IllegalArgumentException();

		this.useMmakeCmd = useMmakeCmd;

		InitCompiler(); // settings
		InitSettings(); // project settings
	}

	// see CompilerHelper class constructor
	public Compiler (Project pObj)
	{
		// set main objects
		super(pObj);
		if (this.projectObject == null)
			throw new IllegalArgumentException();

		this.useMmakeCmd = true;

		InitCompiler(); // settings
		InitSettings(); // project settings
	}

	private void InitCompiler ()
	{
	}

	private void InitSettings ()
	{
		this.mainSourceFile = this.config.get(Compiler.CONFIG_MAIN_SOURCE);

		if ("true".equals(this.config.get(Compiler.CONFIG_REMOTE_BUILD)))
		{
			// remote build enabled

			FileObject rootDir = projectGetRootDirectory();
			this.projectRootPath = (rootDir != null) ? rootDir.getPath() : "";
			this.projectRootParentPath = (rootDir != null) ? rootDir.getParent().getPath() : "";

			this.projectRemoteLogin = this.config.get(Compiler.CONFIG_REMOTE_LOGIN);
			this.cmdSsh   = commandToArray(this.config.get(Compiler.CONFIG_SSH_CMD));
			this.cmdRsync = commandToArray(this.config.get(Compiler.CONFIG_RSYNC_CMD));

			this.projectRemotePath = this.config.get(Compiler.CONFIG_REMOTE_PATH);

			if (this.projectRemoteLogin.isEmpty() ||
				this.cmdSsh.isEmpty() || this.cmdRsync.isEmpty() ||
				this.projectRemotePath.isEmpty()  ||
				this.projectRootPath.isEmpty() || this.projectRootParentPath.isEmpty())
			{
				this.projectRemoteBuild = false;
				return;
			}

			this.projectRemoteBuild = true;

			// paths to upload

			this.projectUploads.add(this.projectRootPath);
			int i = 0;
			String p;
			while ((p = this.config.get(Compiler.CONFIG_REMOTE_UPLOAD + i++)) != null)
				this.projectUploads.add(p);

			//

			this.projectRemoteBins = this.config.get(Compiler.CONFIG_REMOTE_BINS);
		}
	}

	// -----------------------------------------------------------------

	// this method is called when the thread runs
	@Override
	public void run ()
	{
		compile();

		// XXX TODO fix this
		MercuryInterface.compiler_lock.set(false);
	}

	// compile file or project
	public void compile ()
	{
		// check if mmake cmd is empty and we build the project
		// so try to use mmc with main file (from project config) instead
		if (this.useMmakeCmd && MainOptionsPanel.getMercuryMakeCmd().isEmpty() &&
				!this.mainSourceFile.isEmpty())
		{
			// XXX TODO print message
			setDataObject(getFile(this.mainSourceFile.endsWith(".m") ?
									this.mainSourceFile : this.mainSourceFile + ".m"));
			this.useMmakeCmd = !(this.dataObject != null);
		}

		if (this.dataObject != null)
		{
			// save file first if it's been modified (if compiling file)
			// XXX TODO save all modified files if building project
			saveFile(this.dataObject);

			// we generate modules list in build dir to avoid copy all source files
			// but mercury source file name can differ from module name (123.m with m123 module)
			// so get first module name from source file or file name (see filesGetModules)
			ArrayList<String> modules = sourceGetModulesInfo(this.dataObjectFile);
			this.dataObjectModule =
					!modules.isEmpty() ? modules.get(0) : this.dataObjectFile.getName();
		}

		this.compile_internal();
	}

	//
	private void compile_internal ()
	{
		OutputWriter ow = MercuryInterface.getOutputWindow();
		ow.println();

		// get compiler commands and env params

		if (!compileCheckParams(ow))
		{
			ow.close();
			return;
		}

		List<String> compileCmd = compileGetCommand();
		List<String> additionalCmd = commandToArray(MainOptionsPanel.getAdditionalExecuteCmd());
		String localBins = MainOptionsPanel.getBinsPath();

		// prepare to build

		FileObject projectDir = projectGetProjectDirectory();

		if (projectDir != null)
		{
			ow.printf("Found project dir:  %s\n", projectDir.getPath());
		}
		else if (this.dataObject == null)
		{
			// trying build mercury project and no mercury project dir

			ow.printf("project dir '%s' not found, can't build project\n",
						Compiler.PROJECT_DIR);
			return;
		}

		FileObject fileDir = (this.dataObject != null) ? this.dataObjectFile.getParent() : null;
		FileObject sourcesDir = (projectDir != null) ? projectGetSourceDirectory() : fileDir;
		FileObject compileDir =
				(projectDir != null) ? projectGetBuildDirectory() :
					(this.dataObject != null) ? createBuildDirectory(fileDir) :
						null;

		if (compileDir == null)
		{
			// can't create build directory in project root or single file dir
			// XXX TODO print build directory path

			ow.printf("error, can't create build directory\n");
			return;
		}

		// XXX TODO print remote params (not local) if remote build
		ow.printf("Build directory:    %s\n", compileDir.getPath());
		ow.printf("Build command:      %s\n", commandToString(compileCmd));
		if (this.projectRemoteBuild)
			ow.printf("Remote directory:   %s\n", this.projectRemotePath);
		else if (additionalCmd != null)
			ow.printf("Additional command: %s\n", commandToString(additionalCmd));

		ow.println();

		// go!

		String pathReplaceLocal = (this.projectRemoteBuild) ? this.projectRootParentPath : null;
		String pathReplaceRemote = (this.projectRemoteBuild) ? this.projectRemotePath : null;

		try
		{
			// delete files from build dir
			ow.println("Clean up build directory");
			cleanBuildDirectory(compileDir, this.projectRemoteBuild);

			// copy source file if compile file
			// not needed when compiling by module name (see compile())
			//if (this.dataObject != null)
			//{
			//	if(!compileCopySource(ow, compileDir))
			//		return;
			//
			//	ow.println();
			//}

			ow.println();

			// generate additional build files in build dir
			//
			// if we have opened project with 'mercury' subdir:
			//     copy 'mercury/Mercury.modules' to build dir
			//     add all modules names from 'source' dir to 'mercury/Mercury.modules'
			//     copy 'mercury/Mercury.options' to build dir
			//     copy 'mercury/Mmakefile' to build dir
			//
			// otherwise we try to find all these files in current file dir

			// create modules file (Mercury.modules) in build dir
			if (compileGenerateModulesList(ow, compileDir,
											projectDir != null ? projectDir : fileDir, sourcesDir,
											pathReplaceLocal, pathReplaceRemote))
			{
				ow.println();
				ow.println();
			}

			// copy compiler options file (Mercury.options) to build dir
			if (compileGenerateOptions(ow, compileDir,
										projectDir != null ? projectDir : fileDir))
			{
				ow.println();
				ow.println();
			}

			// copy makefile (Mmakefile) to build dir
			if (compileGenerateMakefile(ow, compileDir,
											projectDir != null ? projectDir : fileDir))
			{
				ow.println();
				ow.println();
			}
			//else if (this.dataObject == null)
			else if (this.useMmakeCmd)
			{
				// build project with mmake and no Mmakefile (not needed)

				//ow.printf("'%s' not found in project dir, can't build project\n",
				//			Compiler.PROJECT_MAKEFILE);
				//return;
				ow.printf("WARN! '%s' not found in project dir\n",
							Compiler.PROJECT_MAKEFILE);
				ow.println();
			}

			// upload data (if needed)

			boolean filesPrepeared = true;
			if (this.projectRemoteBuild)
			{
				for (String path : this.projectUploads)
				{
					BufferedReader br =
							uploadExecuteProcess(ow, this.projectRootPath, localBins,
													path, this.projectRemotePath);
					filesPrepeared = (br != null) ? uploadOutputParser(ow, br) : false;
					if (!filesPrepeared)
					{
						ow.printf("'%s' upload error\n", path);
						break;
					}
				}
			}

			// execute mercury command and handling output

			if (filesPrepeared)
			{
				String remoteDir = null;
				String buildDir = compileDir.getPath();

				if (this.projectRemoteBuild)
				{
					remoteDir = buildDir.replaceFirst(pathReplaceLocal, pathReplaceRemote);
					buildDir = this.projectRootPath; // use project root path instead build dir
				}

				BufferedReader br =
						compileExecuteProcess(ow, compileCmd, additionalCmd, localBins,
												buildDir,
												remoteDir, this.projectRemoteBins);
				if (br != null)
					compileOutputParser(ow, br, pathReplaceRemote, pathReplaceLocal);
			}
		}

		finally
		{
			ow.close();
		}
	}

	// construct the Mercury process command array from array of params
	private List<String> compileGetCommand ()
	{
		List<String> compileCmd = new ArrayList<String>();

		// parse compile command and args

		if (!this.useMmakeCmd)
		{
			// use mmc command to compile file or build project

			List<String> cmd = commandToArray(MainOptionsPanel.getMercuryCompilerCmd());
			compileCmd.addAll(cmd);

			//compileCmd.add(this.dataObjectFile.getName()); // filename
			compileCmd.add(this.dataObjectModule); // module name
		}
		else
		{
			// use mmake command to build project

			List<String> cmd = commandToArray(MainOptionsPanel.getMercuryMakeCmd());
			compileCmd.addAll(cmd);
		}

		return compileCmd;
	}

	// check compile options
	private boolean compileCheckParams (OutputWriter ow)
	{
		boolean haveCompiler = true;
		String errStr = null;

		// first see if we have a compiler or make command set

		if (!this.useMmakeCmd &&
			MainOptionsPanel.getMercuryCompilerCmd().isEmpty())
		{
			errStr = "You need to set up the Mercury compiler first.";
			haveCompiler = false;
		}
		else if (this.useMmakeCmd &&
					MainOptionsPanel.getMercuryMakeCmd().isEmpty())
		{
			errStr = "You need to set up the Mercury make or project main file first.";
			haveCompiler = false;
		}

		if (!haveCompiler)
		{
			ow.println(errStr);
			ow.println("In the main menu go to 'Tools / Options / Miscellaneous / Mercury'.");
			ow.println();
			ow.println();

			return false;
		}

		/*
		try
		{
			error_lines = Integer.parseInt(this.mercuryCompileErrorLines);
		}
		catch (NumberFormatException e)
		{
			ow.print(  "Compile Error Lines parameter incorrect.");
			ow.println(" Using default value - 1000.");
			error_lines = 1000;
		}
		*/

		return true;
	}

	// -----------------------------------------------------------------

	// running compile command
	private BufferedReader compileExecuteProcess (OutputWriter ow,
														List<String> compileCmd,
														List<String> additionalCmd, String localBins,
														String buildDir)
	{
		return compileExecuteProcess(ow, compileCmd, additionalCmd, localBins, buildDir,
										null, null);
	}

	private BufferedReader compileExecuteProcess (OutputWriter ow,
														List<String> compileCmd,
														List<String> additionalCmd, String localBins,
														String buildDir,
														String remoteDir, String remoteBins)
	{
		ProcessBuilder procBuilder = new ProcessBuilder();
		// set additional env variables
		String runPath = processUpdatePathEnv(procBuilder, localBins);

		//
		List<String> commandLine;
		if (remoteDir == null)
		{
			// local build

			if (additionalCmd == null)
				commandLine = compileCmd;
			else
			{
				// additional cmd + mercury cmd

				// sh -c "\"/home/user/NetBeansProjects/super test/test.sh\" ; /usr/local/bin/mmc --make main"
				// cmd /C ""c:\olo olo\tools321.bat" & c:\mercury-14.01-vs2013\bin\mercury.bat --make main"

				processFixRunPath(compileCmd, runPath);
				processFixRunPath(additionalCmd, runPath);
				String buildcmd = commandToString(additionalCmd) +
									(Utilities.isWindows() ? " & " : " ; ") +
									commandToString(compileCmd);

				commandLine = new ArrayList<String>();
				//commandLine.addAll(Arrays.asList(cmd));
				commandLine.add(Utilities.isWindows() ? "cmd" : "sh");
				commandLine.add(Utilities.isWindows() ? "/C" : "-c");
				commandLine.add(buildcmd);
			}
		}
		else
		{
			// remote build

			// command to build at remote system:
			// ssh -i ./ssh-key user@server "export PATH=$REMOTE_BINS:$PATH; cd $REMOTE_BUILD_DIR; $BUILD_CMD"

			String remoteCmd = "cd " + remoteDir + "; " + commandToString(compileCmd);
			//String remoteCmd = "cd " + remoteDir + "; pwd";
			if (remoteBins != null && !remoteBins.isEmpty())
				remoteCmd = "export PATH=" + remoteBins + ":$PATH; " + remoteCmd;

			commandLine = new ArrayList<String>();
			commandLine.addAll(this.cmdSsh);
			commandLine.add(   this.projectRemoteLogin);
			commandLine.add(   remoteCmd);
		}

		// set process settings
		File buildDir_f = new File(buildDir);
		procBuilder.command(commandLine);
		procBuilder.redirectErrorStream(true);
		procBuilder.directory(buildDir_f);

		// change proc run cmd (if local build)
		boolean changed = (remoteDir != null || additionalCmd != null ||
							//processFixRunPath(procBuilder, localBins));
							processFixRunPath(procBuilder, runPath));
		if (changed)
			ow.printf("Change build command to '%s'\n", commandToString(procBuilder.command()));

		ow.print("Running build command\n");

		// execute command and handling output
		BufferedReader br = null;
		try
		{
			// execute
			Process process = procBuilder.start();
			InputStream is = process.getInputStream();
			InputStreamReader isr = new InputStreamReader(is);
			br = new BufferedReader(isr);
		}
		catch (IOException ex)
		{
			ow.println("");
			ow.println("Error occurred running Mercury compiler. Check options.");
			ow.println("In the main menu go to 'Tools / Options / Miscellaneous / Mercury'.");
//			Exceptions.printStackTrace(ex);
//			ow.printf("Command string is '%s'.\n", commandGetString(procBuilder.command()));
		}

		return br;
	}

	/*
	 * Sample Mercury output:
	 *
	 * ./hello2.m:005: Warning: source file `./hello2.m' contains module named
	 * ./hello2.m:005:   `hello'.
	 * ./hello2.m:006: Syntax error at token 'interface': operator or `.' expected.
	 * hello.m:001: Warning: interface for module `hello' does not export anything.
	 * For more information, recompile with `-E'.
	 *
	 * ./hello2.m:001: Warning: module should start with a `:- module' declaration.
	 * ./hello2.m:005: Syntax error at token 'module': operator or `.' expected.
	 *
	 * ./hello2.m:005: Warning: source file `./hello2.m' contains module named
	 * ./hello2.m:005:   `hello'.
	 * ./hello2.m:014: Error: state variable !.IO1 is not visible in this context.
	 * ./hello2.m:014: Error: state variable !:IO1 is not visible in this context.
	 * ./hello2.m:014: Warning: reference to uninitialized state variable !.IO1.
	 * ./hello2.m:014: In clause for predicate `hello.main'/2:
	 * ./hello2.m:014:   warning: variable `STATE_VARIABLE_IO1_1' occurs only once in
	 * ./hello2.m:014:   this scope.
	 * ./hello2.m:014: In clause for `main(di, uo)':
	 * ./hello2.m:014:   in argument 2 of call to predicate `io.write_string'/3:
	 * ./hello2.m:014:   unique-mode error: the called procedure would clobber its
	 * ./hello2.m:014:   argument, but variable `IO1' is still live.
	 * For more information, recompile with `-E'.
	 *
	 * /home/xxx/NetBeansProjects/MercurySimpleProject/source/main.m:007: Syntax
	 * /home/xxx/NetBeansProjects/MercurySimpleProject/source/main.m:007:   error at
	 * /home/xxx/NetBeansProjects/MercurySimpleProject/source/main.m:007:   token
	 * /home/xxx/NetBeansProjects/MercurySimpleProject/source/main.m:007:   'main':
	 * /home/xxx/NetBeansProjects/MercurySimpleProject/source/main.m:007:   operator
	 * /home/xxx/NetBeansProjects/MercurySimpleProject/source/main.m:007:   or `.'
	 * /home/xxx/NetBeansProjects/MercurySimpleProject/source/main.m:007:   expected.
	 *
	 * ** Error making `Mercury\int3s\main.int3'.
	 * D:/_projects/_MercurySimpleProject/source/main.m:011: Syntax error at token 'main': operator or `.' expected.
	 *
	 * ** Error: error reading file `123.m' to generate dependencies.
	 *
	 * mercury_compile.exe: cannot find source for module `123' in directories .
	 *
	 * Error: file `D:/_projects/_MercurySimpleProject/source/123.m' contains the
	 *   wrong module.
	 *   Expected module `123', found module `m123'.
	 *
	 * D:/src/mercury/source/hlp_dir.m: In function 'hlp_dir_module3':
	 */

	// get text after filename and source line
	private String lineGetText (String line)
	{
		int fpos = line.indexOf(".m:") + 3;
		int startpos = line.indexOf(":", fpos + 1);
		if (startpos == -1)
			return line.trim();

		StringBuilder text = new StringBuilder();
//		for (int i = 0; i < startpos; i++)
//			text.append(" ");
		text.append(line.substring(startpos + 1).trim());

		return text.toString();
	}

	// get source filename (or null)
	private String lineGetFile (String line)
	{
		String text = null;
		int fpos, spos;

		//
		fpos = line.indexOf(".m:");
		if (fpos != -1)
		{
			text = line.substring(0, fpos + 2).trim();
			return ((text.isEmpty()) ? null : text);
		}

		//
		fpos = line.indexOf(".m'");
		if (fpos == -1)
			return text;

		spos = fpos;
		while (spos > 0 && line.charAt(spos) != '`')
			spos--;
		if (line.charAt(spos) == '`')
			spos++;

		text = line.substring(spos, fpos + 2).trim();
		return ((text.isEmpty()) ? null : text);
	}

	// getting line number from error
	private int lineGetNumber (String line)
	{
		// search for .m:
		// mmc can compile source files only with this extension
		int fpos = line.indexOf(".m:");
		if (fpos == -1)
			return 0;

		fpos += 3;

		int epos = line.indexOf(":", fpos + 1);
		if (epos == -1)
			return 0;

		Character ch;
		StringBuilder numBuf = new StringBuilder();

		for (int i = fpos; i < epos; i++)
		{
			ch = line.charAt(i);
			if (Character.isDigit(ch))
			{
				numBuf.append(ch);
			}
			else
			{
				break;
			}
		}

		if (numBuf.length() == 0)
			return 0;

		int	lineNumber = Integer.parseInt(numBuf.toString());
		return lineNumber;
	}

	// check string for notice, error and etc
	private boolean lineIsNotable (String line)
	{
		boolean result = false;

		/*
		if (line.indexOf("Warning") != -1 ||
			line.indexOf("Error") != -1 ||
			line.indexOf("warning") != -1 ||
			line.indexOf("error") != -1 ||
			line.indexOf("cannot") != -1 ||
			line.indexOf("unexpected") != -1)
		{
			result = true;
		}
		*/

		return result;
	}

	// parse compile command output
	private void compileOutputParser (OutputWriter ow, BufferedReader br)
	{
		compileOutputParser (ow, br, null, null);
	}

	private void compileOutputParser (OutputWriter ow, BufferedReader br,
										String pathReplace, String pathReplaceTo)
	{
		String line;
		StringBuilder simpleline = new StringBuilder();
		StringBuilder errline = new StringBuilder();
		int line_num = 0;
		int errline_num;
		DateFormat dateFormat = new SimpleDateFormat("HH:mm:ss"); // yyyy/MM/dd HH:mm:ss
		boolean log_truncated = false;

		ow.printf("compiling (%s):\n\n", dateFormat.format(new Date()));

		try
		{
			// handling output
			line = br.readLine();
			while (line != null)
			{
				// get text without line number or notable message

				simpleline.setLength(0);
				while (line != null && (line_num = lineGetNumber(line)) == 0 &&
						!lineIsNotable(line) && lineGetFile(line) == null)
				{
					simpleline.append(line).append("\n");
					line = br.readLine();
				}

				if (!simpleline.toString().isEmpty())
				{
					if (!log_truncated &&
							simpleline.indexOf("error log truncated") != -1 &&
							simpleline.indexOf("complete log") != -1)
						log_truncated = true;

					ow.println(simpleline.toString());
				}

				if (line == null)
					continue;

				// process error text

				errline_num = (line_num == 0) ? -1 : line_num;

				errline.setLength(0);
				//errline.append(line).append("\n");
				errline.append(line);

				// read all lines of error message
				while ((line = br.readLine()) != null &&
						((line_num = lineGetNumber(line)) == errline_num ||
							line.startsWith("  ")))
				{
					// XXX TODO uncomment newline when fix outputLineAction
					// file and line detect on multiline error

					//if (errline_num != -1)
					errline.append(" ");
					//else
					//	errline.append("\n");

					errline.append(lineGetText(line));
				}

				String errlineStr = errline.toString();
				if (errlineStr.isEmpty())
					continue;

				// output error message

				if (pathReplace != null)
				{
					String filename = lineGetFile(errlineStr);
					if (filename != null && filename.startsWith(pathReplace))
						errlineStr = errlineStr.replaceFirst(pathReplace, pathReplaceTo);
				}

				ow.println(errlineStr, listener);
				//if (errline_num != -1)
				ow.println();
			} // while
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		ow.printf("\nexit (%s)\n\n", dateFormat.format(new Date()));

		if (log_truncated)
			ow.println("WARN! output log truncated, add '--output-compile-error-lines 1000' to compiler params");
	}

	// output with hotlinks
	private final OutputListener listener = new OutputListener ()
		{
			public void outputLineAction (OutputEvent ev)
			{
				String outputLine = ev.getLine();
				int lineNumber = lineGetNumber(outputLine);
				if (lineNumber <= 0)
					lineNumber = 0;
				else
					lineNumber--;

				// file from error message may be without full path!
				String filename = lineGetFile(outputLine);

				//if (!dataObjectFile.getPath().equals(filename))
				//int columnNumber = parseColumnNumber(outputLine);

				openEditor(filename, lineNumber, true, "Fix me!");
			}

			public void outputLineSelected (OutputEvent ev)
			{
				// let's not do anything special
			}

			public void outputLineCleared (OutputEvent ev)
			{
				// leave it blank, no state to remove
			}
		};

	// -----------------------------------------------------------------

	// running upload command
	// XXX TODO copy/paste from compileExecuteProcess
	private BufferedReader uploadExecuteProcess (OutputWriter ow,
													String runDir, String localBins,
													String path, String uploadPath)
	{
		// project upload to remote system:
		// rsync -avz --delete -e "ssh -i ./ssh-key" $LOCAL_PATH user@server:$REMOTE_PATH

		List<String> cmd = new ArrayList<String>();
		cmd.addAll(this.cmdRsync);
		cmd.add(   "-e");
		cmd.add(   commandToString(this.cmdSsh));
		cmd.add(   path);
		cmd.add(   this.projectRemoteLogin + ":" + uploadPath);
		//String[] cmd = { "pwd" };
		//String[] cmd = { "sh", "-c", "pwd ; echo $PATH" };

		// create process object
		File runDir_f = new File(runDir);
		ProcessBuilder procBuilder = new ProcessBuilder(cmd);
		procBuilder.redirectErrorStream(true);
		procBuilder.directory(runDir_f);

		// set additional env variables
		String runPath = processUpdatePathEnv(procBuilder, localBins);

		// change proc run cmd
		//processFixRunPath(procBuilder, localBins);
		processFixRunPath(procBuilder, runPath);

		ow.printf("Running upload command: %s\n", commandToString(procBuilder.command()));
		//ow.print("Running upload command");

		// execute command and handling output
		BufferedReader br = null;
		try
		{
			// execute
			Process process = procBuilder.start();
			InputStream is = process.getInputStream();
			InputStreamReader isr = new InputStreamReader(is);
			br = new BufferedReader(isr);
		}
		catch (IOException ex)
		{
			ow.println("");
			ow.println("Error occurred running upload command. Check options.");
			ow.println("In the main menu go to 'Tools / Options / Miscellaneous / Mercury'.");
//			Exceptions.printStackTrace(ex);
//			ow.printf("Command string is %s.\n", commandGetString(procBuilder.command()));
		}

		return br;
	}

	// parse upload command output
	//
	// ok
	// sent 1,202,689 bytes  received 1,331 bytes  344,005.71 bytes/sec
	// total size is 2,922,223  speedup is 2.43
	//
	// err
	// sent 18 bytes  received 11 bytes  11.60 bytes/sec
	// total size is 0  speedup is 0.00
	// rsync error: some files/attrs were not transferred (see previous errors) (code 23) at main.c(1183) [sender=3.1.0]
	//
	private boolean uploadOutputParser (OutputWriter ow, BufferedReader br)
	{
		boolean result = true;
		String line;
		DateFormat dateFormat = new SimpleDateFormat("HH:mm:ss"); // yyyy/MM/dd HH:mm:ss

		ow.printf("uploading (%s):\n\n", dateFormat.format(new Date()));

		try
		{
			// handling output
			while ((line = br.readLine()) != null)
			{
				ow.println(line);
				if (line.indexOf("rsync error") != -1)
					result = false;
			}
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		ow.printf("\nexit (%s)\n\n", dateFormat.format(new Date()));

		return result;
	}

	// -----------------------------------------------------------------

	// generate Mercury.modules in build dir
	private boolean compileGenerateModulesList (OutputWriter ow, FileObject buildDir,
														FileObject projectDir, FileObject projectSrcDir)
	{
		return compileGenerateModulesList(ow, buildDir, projectDir, projectSrcDir, null, null);
	}

	private boolean compileGenerateModulesList (OutputWriter ow, FileObject buildDir,
														FileObject projectDir, FileObject sourcesDir,
														String pathReplace, String pathReplaceTo)
	{
		boolean result = false;
		FileObject build_file = null;

		// delete old file

		try
		{
			build_file = buildDir.getFileObject(Compiler.PROJECT_MODSFILE);
			if (build_file != null)
				build_file.delete();
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		// check for Mercury.modules file in mercury project dir and read it

		String modules_file_text = "";
		try
		{
			if (projectDir != null)
			{
				FileObject modules_file = projectDir.getFileObject(Compiler.PROJECT_MODSFILE);
				if (modules_file != null && modules_file.isData())
				{
					ow.printf("Found modules file: %s\n", modules_file.getPath());
					//modules_file.copy(commandDir, modules_file.getName(), modules_file.getExt());
					modules_file_text = modules_file.asText().trim();
				}
			}
		}
		catch (IOException ex)
		{
			ow.println("Error copying modules file to build dir");
//			Exceptions.printStackTrace(ex);
		}

		// getting modules list from project source directory
		// XXX TODO files with same module: main.m old/main.m

		Map<String, String> modules = new HashMap<String, String>();

		if (sourcesDir != null)
		{
			ow.printf("Scanning for modules in '%s'\n", sourcesDir.getPath());
			modules.putAll(filesGetModules(sourcesDir));
		}

		// create Mercury.modules file in build dir

		BufferedWriter bw = null;
		try
		{
			//if (modules_file == null)
				build_file = buildDir.createData(Compiler.PROJECT_MODSFILE);
			//else
			//	build_file = buildDir.getFileObject(this.mercuryModulesFile);

			OutputStream os = build_file.getOutputStream();
			OutputStreamWriter osw = new OutputStreamWriter(os);
			bw = new BufferedWriter(osw);
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		ow.printf("Generating '%s'", build_file.getPath());

		try
		{
			// write user Mercury.modules text
			if (!modules_file_text.isEmpty())
			{
				if (pathReplace != null)
				{
					// XXX TODO on win paths must be in format: C:/sddd/dddd (pathReplace generated by FileObject getPath)
					modules_file_text =
						modules_file_text.replaceAll("\\t" + pathReplace, "\t" + pathReplaceTo);
				}

				bw.write(modules_file_text);
				bw.newLine();
			}

			// write modules info
			Iterator<String> it = modules.keySet().iterator();
			while (it.hasNext())
			{
				String module = it.next();
				String path = modules.get(module);
				if (pathReplace != null && path.startsWith(pathReplace))
					path = path.replaceFirst(pathReplace, pathReplaceTo);

				bw.write(module + "\t" + path);
				bw.newLine();
			}

			bw.close();
			result = true;
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		return result;
	}

	// copy Mercury.options to build dir
	private boolean compileGenerateOptions (OutputWriter ow, FileObject buildDir,
												FileObject projectDir)
	{
		boolean result = false;
		FileObject options_file;

		// delete old file
		try
		{
			options_file = buildDir.getFileObject(Compiler.PROJECT_OPTSFILE);
			if (options_file != null)
				options_file.delete();
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		// check file Mercury.options in mercury dir and read it
		try
		{
			if (projectDir != null)
			{
				options_file = projectDir.getFileObject(Compiler.PROJECT_OPTSFILE);
				if (options_file != null && options_file.isData())
				{
					ow.printf("Found options file: %s\n", options_file.getPath());
					options_file.copy(buildDir, options_file.getName(), options_file.getExt());

					ow.print("File copied to build dir");
					result = true;
				}
			}
		}
		catch (IOException ex)
		{
			ow.println("Error copying options file to build dir");
//			Exceptions.printStackTrace(ex);
		}

		return result;
	}

	// copy Mmakefile to build dir
	private boolean compileGenerateMakefile (OutputWriter ow, FileObject buildDir,
													FileObject projectDir)
	{
		boolean result = false;
		FileObject make_file;

		// delete old file
		try
		{
			make_file = buildDir.getFileObject(Compiler.PROJECT_MAKEFILE);
			if (make_file != null)
				make_file.delete();
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		// check file MMakefile in mercury dir and read it
		try
		{
			if (projectDir != null)
			{
				make_file = projectDir.getFileObject(Compiler.PROJECT_MAKEFILE);
				if (make_file != null && make_file.isData())
				{
					ow.printf("Found makefile: %s\n", make_file.getPath());
					make_file.copy(buildDir, make_file.getName(), make_file.getExt());

					ow.print("File copied to build dir");
					result = true;
				}
			}
		}
		catch (IOException ex)
		{
			ow.println("Error copying makefile to build dir");
//			Exceptions.printStackTrace(ex);
		}

		return result;
	}

	// copy source file to build dir
	private boolean compileCopySource (OutputWriter ow, FileObject buildDir)
	{
		boolean result = false;
		FileObject source_file;

		// delete old file
		try
		{
			source_file = buildDir.getFileObject(this.dataObjectFile.getNameExt());
			if (source_file != null)
				source_file.delete();
		}
		catch (IOException ex)
		{
			Exceptions.printStackTrace(ex);
		}

		// copy source file
		try
		{
			this.dataObjectFile.copy(buildDir, this.dataObjectFile.getName(),
										this.dataObjectFile.getExt());

			ow.printf("File '%s' copied to build dir", this.dataObjectFile.getPath());
			result = true;
		}
		catch (IOException ex)
		{
			ow.printf("Error copying file '%s' to build dir\n",
						this.dataObjectFile.getPath());
//			Exceptions.printStackTrace(ex);
		}

		return result;
	}
}
